//
//  ViewModifiers.swift
//  Do It
//
//  Created by Jim Dovey on 2/4/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI
import Combine

struct DoubleShadow: ViewModifier {
    private var radius: CGFloat
    
    init(_ radius: CGFloat = 10.0) {
        self.radius = radius
    }
    
    func body(content: Content) -> some View {
        content
            .shadow(color: Color.black.opacity(0.1),
                    radius: radius, x: 0, y: radius * 1.2)
            .shadow(color: Color.black.opacity(0.2),
                    radius: max(radius/10, 1), x: 0, y: 1)
    }
}

struct BorderedTextField: ViewModifier {
    func body(content: Content) -> some View {
        content
            .multilineTextAlignment(.center)
            .padding(.vertical, 12)
            .background(
                RoundedRectangle(cornerRadius: 10, style: .continuous)
                    .foregroundColor(.tertiaryFill)
            )
    }
}

// START:HoverAvailability
@available(iOS 13.4, *)
@available(OSX, unavailable)
@available(tvOS, unavailable)
@available(watchOS, unavailable)
struct NicelyHoverable: ViewModifier {
    // <literal:elide> ... </literal:elide>
    private let padding: EdgeInsets
    private let effect: HoverEffect
    @State private var hovering = false
    
    init(_ padding: CGFloat = 8, _ edges: Edge.Set = .all,
         _ effect: HoverEffect = .automatic) {
        self.padding = EdgeInsets(
            top: edges.contains(.top) ? padding : 0,
            leading: edges.contains(.leading) ? padding : 0,
            bottom: edges.contains(.bottom) ? padding : 0,
            trailing: edges.contains(.trailing) ? padding : 0
        )
        self.effect = effect
    }
    
    init(_ insets: EdgeInsets, _ effect: HoverEffect = .automatic) {
        self.padding = insets
        self.effect = effect
    }
    
    func body(content: Content) -> some View {
        content
            .padding(padding)
            .hoverEffect(effect)
            .padding(-padding)
    }
}
// END:HoverAvailability

// START:SceneAvailability
#if canImport(UIKit)
@available(iOS 13.0, tvOS 13.0, *)
@available(OSX, unavailable)
@available(watchOS, unavailable)
struct SceneSessionDismissal: ViewModifier {
    // <literal:elide> ... </literal:elide>
    private let session: UISceneSession
    private let errorHandler: ((Error) -> Void)?
    
    init(_ session: UISceneSession, errorHandler: ((Error) -> Void)? = nil) {
        self.session = session
        self.errorHandler = errorHandler
    }
    
    func body(content: Content) -> some View {
        let button = Button(action: {
            UIApplication.shared
                .requestSceneSessionDestruction(self.session, options: nil,
                                                errorHandler: self.errorHandler)
        }) {
            Text("Done")
                .bold()
        }
        .niceHoverEffect()
        
        return NavigationView {
            content.navigationBarItems(leading: button)
        }
        .navigationViewStyle(StackNavigationViewStyle())
    }
}
#endif
// END:SceneAvailability

// START:SceneDismissalHelper
extension View {
    // END:SceneDismissalHelper
    func doubleShadow(radius: CGFloat = 10) -> some View {
        modifier(DoubleShadow(radius))
    }
    
    @available(iOS 13.4, *)
    @available(OSX, unavailable)
    @available(tvOS, unavailable)
    @available(watchOS, unavailable)
    func niceHoverEffect(_ padding: CGFloat = 8, _ edges: Edge.Set = .all,
                         _ effect: HoverEffect = .automatic) -> some View {
        modifier(NicelyHoverable(padding, edges, effect))
    }
    
    @available(iOS 13.4, *)
    @available(OSX, unavailable)
    @available(tvOS, unavailable)
    @available(watchOS, unavailable)
    func niceHoverEffect(_ insets: EdgeInsets, _ effect: HoverEffect = .automatic) -> some View {
        modifier(NicelyHoverable(insets, effect))
    }
    
    @available(iOS 13.4, *)
    @available(OSX, unavailable)
    @available(tvOS, unavailable)
    @available(watchOS, unavailable)
    func iconHoverEffect(_ color: Color = .primaryGroupedBackground,
                         _ padding: CGFloat = 8,
                         _ effect: HoverEffect = .automatic) -> some View {
        let insets = EdgeInsets(top: padding+6, leading: padding,
                                bottom: padding+6, trailing: padding)
        return modifier(NicelyHoverable(insets, effect))
    }
    
    #if canImport(UIKit)
    @available(iOS 13.0, tvOS 13.0, *)
    @available(OSX, unavailable)
    @available(watchOS, unavailable)
    func dismissingSceneSession(
        _ session: UISceneSession,
        errorHandler: ((Error) -> Void)? = nil
    ) -> some View {
        modifier(SceneSessionDismissal(session, errorHandler: errorHandler))
    }
    #endif
}
// END:SceneDismissalHelper

extension Text {
    func filledPlatter() -> some View {
        modifier(BorderedTextField())
    }
}

extension TextField {
    func filledPlatter() -> some View {
        modifier(BorderedTextField())
    }
}

struct ViewModifiers_Previews: PreviewProvider {
    static var _sampleText: String = "Sample Text"
    static var sampleText: Binding<String> = Binding(
        get: { _sampleText }, set: { _sampleText = $0 })

    static var previews: some View {
        Group {
            Circle()
                .frame(width: 300, height: 300)
                .foregroundColor(.white)
                .modifier(DoubleShadow())

            Circle()
                .frame(width: 300, height: 300)
                .foregroundColor(.white)
                .modifier(DoubleShadow(20))

            Circle()
                .frame(width: 300, height: 300)
                .foregroundColor(.white)
                .modifier(DoubleShadow(6))

            Text("Start Making Things")
                .padding(.horizontal)
                .modifier(BorderedTextField())

            TextField("Title", text: sampleText)
                .modifier(BorderedTextField())
            
            #if os(iOS)
            Button(action: {}) {
                Text("Button").bold()
            }
            .niceHoverEffect()
            
            Button(action: {}) {
                Image(systemName: "info.circle.fill")
                    .imageScale(.large)
            }
            .iconHoverEffect()
            #endif // os(iOS)
        }
        .previewLayout(.fixed(width: 350, height: 350))
    }
}
